<?php

namespace App\Http\Controllers;

use App\Models\LeaveType;
use App\Models\LeaveBalance;
use App\Models\LeaveRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class LeaveController extends Controller
{
    public function index()
    {
        $user = Auth::user();

        $leaveTypes = LeaveType::all();

        // Fetch user's leave balances
        $leaveBalances = LeaveBalance::where('user_id', $user->id)->get()->keyBy('leave_type_id');

        // Fetch user's leave requests
        $leaveRequests = LeaveRequest::where('user_id', $user->id)->orderBy('created_at', 'desc')->get();

        return view('leave.index', compact('user', 'leaveTypes', 'leaveBalances', 'leaveRequests'));
    }

    public function apply(Request $request)
{
    $request->validate([
        'leave_type_id' => 'required|exists:leave_types,id',
        'start_date' => 'required|date|after_or_equal:today',
        'end_date' => 'required|date|after_or_equal:start_date',
        'reason' => 'nullable|string',
        'attachment' => 'nullable|file|max:10240', // 10MB max
    ]);

    try {
        $user = Auth::user();

        // Calculate total leave days (inclusive)
        $totalDays = (strtotime($request->end_date) - strtotime($request->start_date)) / (60*60*24) + 1;

        // Fetch leave balance
        $balance = LeaveBalance::where('user_id', $user->id)
            ->where('leave_type_id', $request->leave_type_id)
            ->first();

        if (!$balance || $balance->balance_days < $totalDays) {
            return redirect()->back()->with('error', 'Insufficient leave balance.');
        }

        // Upload attachment if provided
        $filePath = null;
        if ($request->hasFile('attachment')) {
            $filePath = $request->file('attachment')->store('leave_attachments', 'public');
        }

        // Create leave request
        LeaveRequest::create([
            'user_id' => $user->id,
            'leave_type_id' => $request->leave_type_id,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
            'total_days' => $totalDays,
            'reason' => $request->reason,
            'attachment' => $filePath,
            'status' => 'pending',
        ]);

        return redirect()->back()->with('success', 'Leave request submitted successfully!');

    } catch (\Exception $e) {
        // Log the error for debugging
        \Log::error('Leave apply error: '.$e->getMessage());

        return redirect()->back()->with('error', 'Something went wrong while submitting your leave request. Please try again.');
    }
}

}
