<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    public function login(Request $request)
    {
        // 1. Validate input
        $request->validate([
            'login' => ['required', 'string'],
            'password' => ['required'],
        ]);

        $login = $request->input('login');
        $password = $request->input('password');

        // 2. Determine login field
        $field = filter_var($login, FILTER_VALIDATE_EMAIL)
            ? 'email'
            : 'employee_number';

        // 3. Attempt authentication
        if (! Auth::attempt([$field => $login, 'password' => $password], $request->boolean('remember'))) {
            throw ValidationException::withMessages([
                'login' => 'Invalid login credentials.',
            ]);
        }

        // 4. Regenerate session
        $request->session()->regenerate();

        // 5. Block inactive users
        if (! auth()->user()->employment_status) {
            Auth::logout();

            throw ValidationException::withMessages([
                'login' => 'Your account is inactive. Please contact HR.',
            ]);
        }

        return redirect()->intended('/dashboard');
    }

    public function show()
    {
        return view('auth.forgot-password');
    }

    public function send(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
        ]);

        $user = User::where('email', $request->email)->first();

        if (!$user) {
            return back()->withErrors([
                'email' => 'No account found with this email address'
            ]);
        }

        if (!$user->employment_status) {
            return back()->withErrors([
                'email' => 'This account is deactivated'
            ]);
        }

        // Generate new password
        $newPassword = Str::random(10);

        $user->update([
            'password' => Hash::make($newPassword),
        ]);

        // Send email
        Mail::raw(
            "Your password has been reset.\n\nNew Password: {$newPassword}\n\nPlease change it after logging in.",
            function ($message) use ($user) {
                $message->to($user->email)
                        ->subject('Password Reset');
            }
        );

        return back()->with('success', 'A new password has been sent to your email');
    }

    public function logout(Request $request)
    {
        Auth::logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect('/');
    }
}
